function [out,fmin,parout] = adlpmm (Ffun,Ffun_prox,Gfun,Gfun_prox,Afun,Atfun,lambda,startx,L,par)
% ADLPMM employs the Alternating directions linearized proximal method of multipliers
% for solving the problem min{  f(x) + lambda* g( A(x)) }
%
% Underlying assumptions:
% All functions are convex
% f is is proper closed and proximable
% g ia is proper closed and proximable
% A is a linear transformation
% lambda is a positive scalar
% ====================================================
% Usage:
% out               = ADLPMM (Ffun,Ffun_prox,Gfun,Gfun_prox,Afun,Atfun,lambda,startx,[L],[par])
% [out,fmin]        = ADLPMM (Ffun,Ffun_prox,Gfun,Gfun_prox,Afun,Atfun,lambda,startx,[L],[par])
% [out,fmin,parout] = ADLPMM (Ffun,Ffun_prox,Gfun,Gfun_prox,Afun,Atfun,lambda,startx,[L],[par])
% ====================================================
% Input:
% Ffun        - function handle for the function f
% Ffun_prox   - function handle for the proximal mapping of f times a postive constant
% Gfun        - function handle for the function g
% Gfun_prox   - function handle for the proximal mapping of g times a postive constant
% Afun        - function handle for the linear transformation of A
% Atfun       - function handle for the adjoint of the linear transformation of A
% lambda      - positive scalar penalty for the function g
% startx      - starting vector
% L           - positive constant; should be set to the squared norm of A 
%               (if the parameter is not given, then the squared norm of A is computed inefficiently)
% par         - struct which contains different values required for the operation of SFISTA
% Fields of par:
%       max_iter         - maximal number of iterations [default: 1000]
%       eco_flag         - true if economic version (without calculating objective function
%                          values) should run, otherwise false [default: false]
%       print_flag       - true if internal printing should take place, otherwise false [default: true]
%       real_valued_flag - true if g is assumed to be a real-valued function [default: false]
%                          when the flag is set to true, only improved function values are printed
%       feas_check_flag  - true if feasibly violation should be printed [default: true]
%                          If real_valued_flag is true, then feas_check_flag is set to false
%       rho              - positive parameter [default: 1]
%       eps              - stopping criteria tolerance (the method stops when the
%                          norm of the difference between consecutive iterates is < eps) [default: 1e-5]
% ====================================================
% Output:
% out    - optimal solution (up to a tolerance)
% fmin   - optimal value (up to a tolerance)
% parout - a struct containing additional information related to the convergence. The fields of parout are:
%    iterNum   - number of performed iterations
%    funValVec - vector of all function values generated by the method
%    FeasVal   - vecror of all feasibly violation values (relevant only when feas_check_flag is true)
%    dualVec   - dual optimal vector

% This file is part of the FOM package - a collection of first order methods for solving convex optimization problems
% Copyright (C) 2017 Amir and Nili Beck
% 
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.


if ((nargin < 8) || ( isa (Ffun_prox,'function_handle')~= 1) || ( isa (Gfun_prox,'function_handle')~= 1) || ...
        ( isa (Afun,'function_handle')~= 1) || ( isa (Atfun,'function_handle')~= 1))
    error ('usage: adlpmm (Ffun,Ffun_prox,Gfun,Gfun_prox,Afun,Atfun,lambda,startx,[L],[par]) ') ;
end


%setting default values
if (nargin < 10)
    par = struct() ;
end

if ( isempty(Ffun) || isempty(Gfun) || ~isa(Ffun,'function_handle') || ~isa(Gfun,'function_handle') )
    if (nargout > 1)
        error('When FFun or Gfun is not a function handle usage: out = adlpmm(Ffun,Ffun_prox,Gfun,Gfun_prox,Afun,Atfun,labmda,startx,[L],[par])')
    else
        if ( ~isfield (par,'eco_flag') || (~par.eco_flag))
            fprintf('Ffun or Gfun is not a function handle, changing to economical mode\n') ;
            par.eco_flag = true ;
            par.print_flag = false ;
        end
    end
end

par = initialize_par_for_adlpmm(par) ;

%starting value for L
A_start_value = Afun(startx)  ;
if ((nargin < 9) || (isempty(L) == 1))
    L = startL_for_adlpmm (A_start_value) ;
end
if (par.print_flag)
    fprintf('using L = %f \n',L) ;
end


%constant values

max_iter = par.max_iter ;
print_flag = par.print_flag ;
eco_flag = par.eco_flag ;
eps = par.eps ;
rho = par.rho ;
real_valued_flag = par.real_valued_flag ;
feas_check_flag = par.feas_check_flag ;


%starting values
iternu=0; %iteration number
prevx = startx ;
A_prev_value = A_start_value ;
prevz = A_prev_value ;
prevy = zeros(size(A_prev_value)) ;
done = false ;
best_vector = startx ;
best_dual = prevy ;
best_value = Ffun(startx) + lambda * Gfun(A_start_value) ;

if (nargout == 3)
    parout = struct ;
    parout.iterNum = 0 ;
    parout.funValVec = [] ;
    if (feas_check_flag)
        parout.feasVal = [] ;
    end
end

if (print_flag)
    fprintf('*********************\n');
    fprintf('adlpmm\n') ;
    fprintf('*********************\n')
    fprintf('#iter       \tfun. val.')
    if (feas_check_flag)
        fprintf('     \t feas. viol. ')
    end
    fprintf('\n') ;
    
end


%the iterations
while ((iternu < max_iter) && (~done))
    iternu = iternu + 1;
    
    newx = Ffun_prox(prevx- 1/L* Atfun(A_prev_value-prevz+1/rho*prevy),1/(rho*L)) ;
    A_new_value = Afun(newx) ;
    newz = Gfun_prox(A_new_value   + 1/rho*prevy,lambda/rho) ;
    newy = prevy + rho*(A_new_value  - newz )  ;
    
    if (~eco_flag)
        if (real_valued_flag)
            fun_val = Ffun(newx)+lambda*Gfun(A_new_value);
        else
            fun_val = Ffun(newx)+lambda*Gfun(newz);
        end
        if (~real_valued_flag || (fun_val < best_value))
            best_value = fun_val ;
            best_vector = newx ;
            best_dual = newy ;
            if (feas_check_flag)
                feas_val = norm(A_new_value-newz,'fro') ;
            end
            if (print_flag)
                fprintf('%6d \t    %12f \t',iternu,fun_val) ;
                if (feas_check_flag)
                    fprintf('%14g \t',feas_val) ;
                end
                fprintf('\n') ;
            end
            
        end
        if (nargout == 3)
            parout.funValVec = [ parout.funValVec ; fun_val] ;
            if (feas_check_flag)
                parout.feasVal = [parout.feasVal ; feas_val ] ;
            end
        end
    end
    
    if (iternu > 1)
        if (norm(newx - prevx,'fro') < eps)
            
            done = true ;
        end
    end
    
    prevx = newx ;
    A_prev_value = A_new_value ;
    prevy = newy ;
    prevz = newz ;
    
end

if (done && print_flag)
    fprintf('Stopping because the norm of the difference between consecutive iterates is too small\n')
end

out = best_vector;

if ((nargout >1) || (print_flag))
    if (~eco_flag)
        fmin = best_value ;
    else
        fmin = Ffun(out)+lambda*Gfun(Afun(out)) ;
    end
end

if (nargout == 3)
    parout.iterNum = iternu ;
    parout.dualVec = best_dual ;
end

if (print_flag)
    fprintf('----------------------------------\n') ;
    fprintf('Optimal value = %15f \n',fmin) ;
end


    function L = startL_for_adlpmm(outstart)
        % L is supposed to be equal to norm(A)^2
        inlen = length(startx(:)) ;
        outlen = length (outstart(:)) ;
        ineye = eye(inlen) ;
        outeye = zeros(outlen,inlen);
        
        for i = 1: inlen
            temp = reshape(ineye(:,i),size(startx)) ;
            outeye(:,i) = reshape(Afun(temp),outlen,1) ;
            
        end
        
        L = norm(outeye,2)^2 ;
    end


end



